<?php

require_once __DIR__ . '/../../config/database.php';

class AuthController {
    private $conn;

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }

    public function register($displayName, $email, $password, $passwordConfirm) {
        $errors = [];

        // Validation des entrées
        if (empty($displayName) || empty($email) || empty($password) || empty($passwordConfirm)) {
            $errors[] = "Tous les champs sont requis.";
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = "Format d'email invalide.";
        }

        if (strlen($password) < 8) {
            $errors[] = "Le mot de passe doit contenir au moins 8 caractères.";
        }

        if ($password !== $passwordConfirm) {
            $errors[] = "Les mots de passe ne correspondent pas.";
        }

        // Vérifier si l'email existe déjà
        if (empty($errors)) {
            try {
                $query = "SELECT id FROM users WHERE email = :email";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':email', $email);
                $stmt->execute();
                if ($stmt->rowCount() > 0) {
                    $errors[] = "Cet email est déjà utilisé.";
                }
            } catch (PDOException $e) {
                error_log("Database error in email check: " . $e->getMessage());
                $errors[] = "Une erreur de base de données est survenue lors de la vérification de l'email.";
            }
        }

        if (empty($errors)) {
            // Hasher le mot de passe
            $passwordHash = password_hash($password, PASSWORD_DEFAULT);
            // L'ID du rôle 'client' est 2 (admin=1, client=2 basé sur l'ordre d'insertion dans schema.sql)
            $roleId = 2; 

            try {
                $this->conn->beginTransaction();

                $query = "INSERT INTO users (display_name, email, password_hash, role_id) 
                          VALUES (:display_name, :email, :password_hash, :role_id)";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':display_name', $displayName);
                $stmt->bindParam(':email', $email);
                $stmt->bindParam(':password_hash', $passwordHash);
                $stmt->bindParam(':role_id', $roleId, PDO::PARAM_INT);
                
                if ($stmt->execute()) {
                    $userId = $this->conn->lastInsertId();

                    // Log de l'inscription
                    $log_query = "INSERT INTO logs (user_id, action, entity, entity_id, ip_address, details) 
                                  VALUES (:user_id, 'create', 'user', :entity_id, :ip, :details)";
                    $log_stmt = $this->conn->prepare($log_query);
                    
                    $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'UNKNOWN';
                    $details = "User registered: " . $email;
                    
                    $log_stmt->bindParam(':user_id', $userId, PDO::PARAM_INT);
                    $log_stmt->bindParam(':entity_id', $userId, PDO::PARAM_INT);
                    $log_stmt->bindParam(':ip', $ip_address);
                    $log_stmt->bindParam(':details', $details);
                    $log_stmt->execute();
                    
                    $this->conn->commit();
                    return ['success' => true];
                } else {
                    $this->conn->rollBack();
                    $errors[] = "Erreur lors de la création du compte. Veuillez réessayer.";
                }
            } catch (PDOException $e) {
                $this->conn->rollBack();
                error_log("Registration error: " . $e->getMessage());
                $errors[] = "Une erreur technique est survenue lors de l'inscription. Veuillez réessayer plus tard.";
            }
        }
        return ['success' => false, 'errors' => $errors];
    }
}
?>
